import { storeToRefs } from 'pinia';
import { useRouting } from '../../desktop/composables/useRouting.ts';
import { chatService } from '@main/common/services/external-services.ts';
import { useAppStore } from '@main/common/store/app.store.ts';
import { areAddressesEqual, toCanonicalAddress } from '@shared/address-utils.ts';
import type { ChatIdObj } from '~/asmail-msgs.types.ts';
import type { IncomingCallCmdArg, MainWindowCommand, OpenChatCmdArg } from '~/chat-commands.types.ts';

export function useCommandHandler() {
  const { user: ownAddr } = storeToRefs(useAppStore());
  const { goToChatsRoute, goToChatRoute } = useRouting();

  async function findOneToOneChatWithPeer(peerAddr: string): Promise<ChatIdObj | undefined> {
    const chats = await chatService.getChatList();
    const peerCAddr = toCanonicalAddress(peerAddr);
    for (const { isGroupChat, chatId } of chats) {
      if (!isGroupChat && (chatId === peerCAddr)) {
        return { isGroupChat, chatId };
      }
    }
  }

  async function openChatWith(cmdArg: OpenChatCmdArg): Promise<void> {
    const peerAddress = cmdArg?.peerAddress;
    if ((typeof peerAddress !== 'string') || !peerAddress) {
      console.error(`Invalid peer address passed in open chat command`);
      return;
    }

    if (areAddressesEqual(peerAddress, ownAddr.value)) {
      await goToChatsRoute();
      return;
    }

    const chatId = cmdArg.chatId
      ? cmdArg.chatId
      : await findOneToOneChatWithPeer(peerAddress);

    if (chatId) {
      await goToChatRoute(chatId);
    } else {
      await goToChatsRoute({ createNew: true });
    }
  }

  async function showIncomingCall({ chatId, peerAddress: callingPeer }: IncomingCallCmdArg): Promise<void> {
    await goToChatRoute(chatId, { callingPeer });
  }

  async function process({ cmd, params }: web3n.shell.commands.CmdParams): Promise<void> {
    try {
      if ((cmd as MainWindowCommand) === 'open-chat-with') {
        await openChatWith(params[0]);
      } else if ((cmd as MainWindowCommand) === 'incoming-call') {
        await showIncomingCall(params[0]);
      } else {
        console.error(`🫤 Unknown/unimplemented command`, cmd, `with parameters`, params);
      }
    } catch (err) {
      console.error(`Error occurred while handing command`, err);
      w3n.log('error', `Error occurred while handing command`, err);
    }
  }

  async function start(): Promise<void> {
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    const unsub = w3n.shell!.watchStartCmds!({
      next: cmdParams => {
        process(cmdParams);
      },
      error: err => console.error(`Error in listening to commands for chat app:`, err),
      complete: () => console.info(`Listening to commands for chat app is closed by platform side.`),
    });

    const startCmd = await w3n.shell!.getStartedCmd!();
    if (startCmd) {
      await process(startCmd);
    }
  }

  return { start };
}
