/*
Copyright (C) 2024 - 2025 3NSoft Inc.

This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation, either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with
this program. If not, see <http://www.gnu.org/licenses/>.
*/
import { computed, inject, ref } from 'vue';
import { defineStore, storeToRefs } from 'pinia';
import keyBy from 'lodash/keyBy';
import { VUEBUS_KEY, VueBusPlugin } from '@v1nt1248/3nclient-lib/plugins';
import { DbRecordException } from '@bg/utils/exceptions.ts';
import { includesAddress, toCanonicalAddress } from '@shared/address-utils.ts';
import { areChatIdsEqual } from '@shared/chat-ids.ts';
import { chatService, fileLinkStoreSrv } from '@main/common/services/external-services.ts';
import { toRO } from '@main/common/utils/readonly.ts';
import { useAppStore } from '@main/common/store/app.store.ts';
import { useChatsStore } from '@main/common/store/chats.store.ts';
import type { ChatIdObj, ChatMessageId } from '~/asmail-msgs.types.ts';
import type { ChatListItemView, ChatMessageAttachmentsInfo, ChatMessageView, GroupChatView } from '~/chat.types.ts';
import type { AppGlobalEvents } from '~/app.types.ts';
import type { RelatedMessage } from '~/asmail-msgs.types.ts';
import type { AddressCheckResult, ChatMessageEvent } from '~/services.types.ts';

type ReadonlyFile = web3n.files.ReadonlyFile;

export interface ChatException extends web3n.RuntimeException {
  type: 'chat';
  failedAddresses?: {
    addr: string;
    check?: AddressCheckResult;
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    exc?: any;
  }[];
}

function makeChatException(fields: Partial<ChatException>): ChatException {
  return {
    ...fields,
    runtimeException: true,
    type: 'chat',
  };
}

export const useChatStore = defineStore('chat', () => {
  const { user: me } = storeToRefs(useAppStore());

  const chatsStore = useChatsStore();
  const { refreshChatList, getChatView, refreshChatViewData } = chatsStore;
  const $emitter = inject<VueBusPlugin<AppGlobalEvents>>(VUEBUS_KEY)!.$emitter;

  const currentChatId = ref<ChatIdObj>();
  const objOfCurrentChatMessages = ref<Record<string, ChatMessageView>>({});

  const currentChat = computed(() => currentChatId.value ? getChatView(currentChatId.value) : null);
  const currentChatMessages = computed(() => Object.values(objOfCurrentChatMessages.value)
    .sort((a, b) => a.timestamp - b.timestamp));

  const isAdminOfGroupChat = computed(() => {
    const chat = currentChat.value;
    return ((chat && chat.isGroupChat) ? chat.admins.includes(me.value) : false);
  });

  function isMemberAdminOfGroupChat(user: string): boolean {
    return !!(currentChat.value?.isGroupChat && (currentChat.value?.admins || []).includes(user));
  }

  async function absorbMessageUpdateEvent(event: ChatMessageEvent): Promise<void> {
    console.log('### HANDLE MESSAGE EVENT ### ', event);
    switch (event.event) {
      case 'added':
        return handleAddedMsg(event.msg);
      case 'updated':
        return handleUpdatedMsg(event.msg);
      case 'removed':
        return handleRemovedMsg(event.msgId);
      default:
        console.log(`Unknown update event from ChatService:`, event);
        break;
    }
  }

  async function handleAddedMsg(msg: ChatMessageView): Promise<void> {
    if (areChatIdsEqual(currentChatId.value, msg.chatId)) {
      if (!objOfCurrentChatMessages.value[msg.chatMessageId]) {
        objOfCurrentChatMessages.value[msg.chatMessageId] = msg;
      }
    }
    await refreshChatViewData(msg.chatId);
  }

  async function handleUpdatedMsg(msg: ChatMessageView): Promise<void> {
    if (areChatIdsEqual(currentChatId.value, msg.chatId)) {
      objOfCurrentChatMessages.value[msg.chatMessageId] = msg;
    }
    await refreshChatViewData(msg.chatId);
  }

  async function handleRemovedMsg(msg: ChatMessageId): Promise<void> {
    if (areChatIdsEqual(currentChatId.value, msg.chatId)) {
      delete objOfCurrentChatMessages.value[msg.chatMessageId];
    }
    await refreshChatViewData(msg.chatId);
  }

  async function setChatAndFetchMessages(chatId: ChatIdObj) {
    if (currentChatId.value?.chatId === chatId.chatId) {
      return;
    }
    if (!getChatView(chatId)) {
      await refreshChatList();
      if (!getChatView(chatId)) {
        throw new Error(`Chat is not found with id ${JSON.stringify(chatId)}`);
      }
    }
    currentChatId.value = chatId;
    await fetchMessages();
  }

  function resetCurrentChat(): void {
    currentChatId.value = undefined;
    objOfCurrentChatMessages.value = {};
  }

  async function fetchMessages(): Promise<void> {
    if (!currentChatId.value) {
      objOfCurrentChatMessages.value = {};
      return;
    }

    objOfCurrentChatMessages.value = keyBy(await chatService.getMessagesByChat(currentChatId.value), 'chatMessageId');
  }

  function getMessageInCurrentChat(chatMsgId: string): ChatMessageView | undefined {
    if (!chatMsgId) {
      return;
    }
    return objOfCurrentChatMessages.value[chatMsgId];
  }

  function ensureCurrentChatIsSet(expectedChatId?: ChatIdObj): void {
    let sure = false;
    if (currentChatId.value) {
      sure = expectedChatId ? areChatIdsEqual(currentChatId.value, expectedChatId) : true;
    }

    if (!sure) {
      throw new Error(`Chat referenced by it ${expectedChatId} is not set current in store`);
    }
  }

  async function deleteMessageInChat(chatMsgId: string, deleteForEveryone?: boolean): Promise<void> {
    const message = getMessageInCurrentChat(chatMsgId);
    if (!message) {
      return;
    }

    try {
      const { chatId, chatMessageId } = message;
      await chatService.deleteMessage({ chatId, chatMessageId }, !!deleteForEveryone);
      await refreshChatViewData(chatId);
    } catch (err) {
      if ((err as DbRecordException).chatNotFound) {
        await refreshChatList();
      } else {
        await fetchMessages();
      }
    }
  }

  async function deleteAllMessagesInChat(
    chatId: ChatIdObj, deleteForEveryone?: boolean,
  ): Promise<void> {
    ensureCurrentChatIsSet(chatId);
    await chatService.deleteMessagesInChat(chatId, !!deleteForEveryone);
    await fetchMessages();
  }

  async function sendMessageInChat(
    { chatId, text, files, relatedMessage, withoutCurrentChatCheck }: {
      chatId: ChatIdObj,
      text: string,
      files: web3n.files.ReadonlyFile[] | undefined,
      relatedMessage: RelatedMessage | undefined,
      withoutCurrentChatCheck?: boolean,
    }) {
    !withoutCurrentChatCheck && ensureCurrentChatIsSet(chatId);
    await chatService.sendRegularMessage(chatId, text, files ?? [], relatedMessage);
    $emitter.emit('send:message', { chatId });
  }

  async function renameChat(
    chat: ChatListItemView, newChatName: string,
  ): Promise<void> {
    const chatId = { isGroupChat: chat.isGroupChat, chatId: chat.chatId };
    ensureCurrentChatIsSet(chatId);
    await chatService.renameChat(chatId, newChatName);
  }

  async function deleteChat(chatId: ChatIdObj, withReset?: boolean): Promise<void> {
    ensureCurrentChatIsSet(chatId);
    withReset && resetCurrentChat();
    await chatService.deleteChat(chatId);
  }

  async function updateGroupMembers(
    chatId: string,
    newMembers: Record<string, { hasAccepted: boolean }>,
  ): Promise<void> {
    const chatIdObj = { isGroupChat: true, chatId };
    ensureCurrentChatIsSet(chatIdObj);

    if (!includesAddress(Object.keys(newMembers), me.value)) {
      throw new Error(`This function can't remove self from members. Own address should be among new members.`);
    }

    const { members } = (currentChat.value as GroupChatView);
    const membersToDelete = Object.keys(newMembers).reduce((res, addr) => {
      if (!includesAddress(Object.keys(newMembers), addr)) {
        res[addr] = newMembers[addr];
      }

      return res;
    }, {} as Record<string, { hasAccepted: boolean }>);

    const membersUntouched = Object.keys(newMembers).reduce((res, addr) => {
      if (!membersToDelete[addr]) {
        res[addr] = newMembers[addr];
      }

      return res;
    }, {} as Record<string, { hasAccepted: boolean }>);

    const membersToAdd = Object.keys(newMembers).reduce((res, addr) => {
      if (!includesAddress(Object.keys(members), addr)) {
        res[addr] = newMembers[addr];
      }

      return res;
    }, {} as Record<string, { hasAccepted: boolean }>);

    await ensureAllAddressesExist(Object.keys(membersToAdd).filter(
      member => (toCanonicalAddress(member) !== toCanonicalAddress(me.value)),
    ));

    if (Object.keys(membersToDelete).length === 0 && Object.keys(membersToAdd).length === 0) {
      return;
    }
    const membersAfterUpdate = { ...membersUntouched, ...membersToAdd };

    await chatService.updateGroupMembers(
      chatIdObj,
      { membersToDelete, membersToAdd, membersAfterUpdate },
    );
  }

  async function ensureAllAddressesExist(members: string[]): Promise<void> {
    const checks = await Promise.all(members.map(async addr => {
      const {
        check, exc,
      } = await chatService.checkAddressExistenceForASMail(addr).then(
        check => ({ check, exc: undefined }),
        exc => ({ check: undefined, exc }),
      );
      return { addr, check, exc };
    }));
    const failedAddresses = checks.filter(({ check }) => (check !== 'found'));
    if (failedAddresses.length > 0) {
      throw makeChatException({ failedAddresses });
    }
  }

  // eslint-disable-next-line @typescript-eslint/no-unused-vars
  async function updateGroupAdmins(chatId: string, newAdmins: string[]): Promise<void> {
    // TODO
    throw new Error(`chatStore updateGroupAdmins() function is not implemented, yet`);
  }

  async function markMessageAsRead(
    chatId: ChatIdObj, chatMessageId: string,
  ): Promise<void> {
    await chatService.markMessageAsReadNotifyingSender({
      chatId, chatMessageId,
    });
  }

  async function getChatMessage(
    id: ChatMessageId,
  ): Promise<ChatMessageView | undefined> {
    return await chatService.getMessage(id);
  }

  async function getMessageAttachments(
    info: ChatMessageAttachmentsInfo[],
    incomingMsgId?: string,
  ): Promise<Record<string, ReadonlyFile>> {
    const files: Record<string, ReadonlyFile> = {};
    if (incomingMsgId) {
      const msg = await chatService.getIncomingMessage(incomingMsgId);

      if (msg && msg.attachments) {
        for (const { name } of info) {
          files[name] = await msg.attachments.readonlyFile(name);
        }
      }
    } else {
      for (const { id, name } of info) {
        if (id) {
          const file = await fileLinkStoreSrv.getFile(id);
          if (file) {
            files[name] = file;
          }
        }
      }
    }
    return files;
  }


  return {
    currentChatId: toRO(currentChatId),
    objOfCurrentChatMessages: toRO(objOfCurrentChatMessages),
    currentChatMessages: toRO(currentChatMessages),

    currentChat,
    isAdminOfGroupChat,

    isMemberAdminOfGroupChat,
    absorbMessageUpdateEvent,
    setChatAndFetchMessages,
    resetCurrentChat,
    deleteAllMessagesInChat,
    sendMessageInChat,
    markMessageAsRead,
    getChatMessage,
    getMessageAttachments,
    deleteMessageInChat,
    renameChat,
    deleteChat,
    updateGroupMembers,
    updateGroupAdmins,
  };
});

export type ChatStore = ReturnType<typeof useChatStore>;
