/*
Copyright (C) 2024 - 2025 3NSoft Inc.

This program is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation, either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with
this program. If not, see <http://www.gnu.org/licenses/>.
*/
import { computed, ref } from 'vue';
import { defineStore, storeToRefs } from 'pinia';
import { useRoute, useRouter } from 'vue-router';
import cloneDeep from 'lodash/cloneDeep';
import size from 'lodash/size';
import { useAppStore } from '@main/common/store/app.store.ts';
import { type ChatStore, useChatStore } from './chat.store.ts';
import type { ChatIdObj, ChatMessageId } from '~/asmail-msgs.types.ts';
import {
  ChatListItemView,
  IncomingCallCmdArg,
  AddressCheckResult,
  ChatEvent,
  UpdateEvent,
} from '~/index.ts';
import { getChatName } from '@main/common/utils/chat-ui.helper.ts';
import { chatService, videoOpenerSrv } from '@main/common/services/external-services.ts';
import { areChatIdsEqual } from '@shared/chat-ids.ts';
import { randomStr } from '@shared/randomStr.ts';
import { SingleProc } from '@shared/processes/single.ts';

export type ChatsStore = ReturnType<typeof useChatsStore>;

export interface ChatCreationException extends web3n.RuntimeException {
  type: 'chat-creation';
  failedAddresses: {
    addr: string;
    check?: AddressCheckResult;
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    exc?: any;
  }[];
}

export const useChatsStore = defineStore('chats', () => {
  const route = useRoute();
  const router = useRouter();

  const { user: me } = storeToRefs(useAppStore());

  const chatList = ref<ChatListItemView[]>([]);
  const newChatDialogFlag = ref(false);
  const incomingCalls = ref<IncomingCallCmdArg[]>([]);

  const chatListSortedByTime = computed(() => chatList.value
    .map(c => ({
      ...c,
      displayName: getChatName(c),
    }))
    .sort((a, b) => {
      const tA = a.lastMsg?.timestamp || a.createdAt;
      const tB = b.lastMsg?.timestamp || b.createdAt;
      return tB - tA;
    }),
  );

  function getChatView(chatId: ChatIdObj): ChatListItemView | undefined {
    return chatList.value.find(cv => areChatIdsEqual(cv, chatId));
  }

  async function refreshChatViewData(chatId: ChatIdObj) {
    const chatViewData = await chatService.getChat(chatId);

    if (chatViewData) {
      const ind = findIndexOfChatInCurrentList(chatViewData);
      if (ind >= 0) {
        const callStart = chatList.value[ind].callStart;
        const incomingCall = chatList.value[ind].incomingCall;

        chatList.value[ind] = {
          ...chatViewData,
          callStart,
          incomingCall,
        };
      }
    }
  }

  function clearIncomingCallsData() {
    incomingCalls.value = [];
  }

  async function createNewOneToOneChat(
    name: string,
    peerAddr: string,
    ownName?: string,
  ): Promise<ChatIdObj | undefined> {
    try {
      return await chatService.createOneToOneChat({ name, peerAddr, ownName });
    } catch (error: unknown) {
      // TODO Maybe it should show a notification about this.
      w3n.log('error', 'Error creating a one to one chat. ', error);
    }
  }

  async function createNewGroupChat(
    name: string,
    groupMembers: Record<string, { hasAccepted: boolean }>,
  ): Promise<ChatIdObj | undefined> {
    try {
      return await chatService.createGroupChat({
        name,
        chatId: randomStr(20),
        members: groupMembers,
      });
    } catch (error: unknown) {
      // TODO Maybe it should show a notification about this.
      w3n.log('error', 'Error creating a group chat. ', error);
    }
  }

  async function acceptChatInvitation({ chatId, chatMessageId }: ChatMessageId, ownName?: string): Promise<void> {
    try {
      if (!ownName) {
        ownName = me.value.substring(0, me.value.indexOf('@'));
      }
      return await chatService.acceptChatInvitation(chatId, chatMessageId, ownName);
    } catch (err) {
      console.error(`Accepting chat invite failed with`, err);
      throw err;
    }
  }

  async function refreshChatList() {
    chatList.value = await chatService.getChatList();
    resetRouteIfItPointsToRemovedChat();
  }

  function resetRouteIfItPointsToRemovedChat(): void {
    const { chatId } = route.params as { chatId: string };
    if (chatId) {
      const foundChat = chatList.value.find(c => (c.chatId === chatId));
      if (!foundChat) {
        router.push({ name: 'chats' });
      }
    }
  }

  function findIndexOfChatInCurrentList(chatId: ChatIdObj): number {
    return chatList.value.findIndex(
      c => areChatIdsEqual(c, chatId),
    );
  }

  async function updateChatItemInList(chatId: ChatIdObj, value: Partial<ChatListItemView>) {
    let chatInd = findIndexOfChatInCurrentList(chatId);
    if (chatInd < 0) {
      await refreshChatList();
      chatInd = findIndexOfChatInCurrentList(chatId);
      if (chatInd < 0) {
        console.error(`The chat with chatId ${chatId.chatId} does not exist`);
        return;
      }
    }

    const currentValue: ChatListItemView = cloneDeep(chatList.value[chatInd]);
    // @ts-expect-error
    chatList.value[chatInd] = {
      ...currentValue,
      ...value,
    };
  }

  const updatesQueue: UpdateEvent[] = [];
  const updatesProc = new SingleProc();

  async function processQueuedUpdateEvents(): Promise<void> {
    while (updatesQueue.length > 0) {
      const event = updatesQueue.pop()!;
      try {
        switch (event.updatedEntityType) {
          case 'chat':
            return absorbChatUpdateEvent(event);
          case 'message':
            return absorbMessageUpdateEvent(event);
          default:
            console.info(`Unknown update event from ChatService:`, event);
            break;
        }
      } catch (err) {
        console.error(err);
      }
    }
  }

  async function absorbChatUpdateEvent(event: ChatEvent): Promise<void> {
    switch (event.event) {
      case 'updated': {
        const { chat } = event;
        const chatInd = findIndexOfChatInCurrentList(chat);
        if (chatInd < 0) {
          await refreshChatList();
        } else {
          const callStart = chatList.value[chatInd].callStart;
          const incomingCall = chatList.value[chatInd].incomingCall;

          chatList.value[chatInd] = {
            ...chat,
            callStart,
            incomingCall,
          };
        }
        break;
      }

      case 'added': {
        const { chat } = event;
        const chatInd = findIndexOfChatInCurrentList(chat);
        if (chatInd < 0) {
          chatList.value.unshift(chat);
        } else {
          await refreshChatList();
        }
        break;
      }

      case 'removed': {
        const { chatId } = event;
        const chatInd = findIndexOfChatInCurrentList(chatId);
        if (chatInd >= 0) {
          chatList.value.splice(chatInd, 1);
        }
        resetRouteIfItPointsToRemovedChat();
        break;
      }

      default:
        throw Error(`Unknown chat event: ${event.event}`);
    }
  }

  function setChatDialogFlag(value: boolean) {
    newChatDialogFlag.value = value;
  }

  function setIncomingCallsData(cmd: IncomingCallCmdArg) {
    incomingCalls.value.push(cmd);
  }

  let absorbMessageUpdateEvent: ChatStore['absorbMessageUpdateEvent'];
  let stopMessagesProcessing: (() => void) | undefined = undefined;
  let stopVideoCallsWatching: (() => void) | undefined = undefined;

  async function initialize() {
    const chatStore = useChatStore();
    absorbMessageUpdateEvent = chatStore.absorbMessageUpdateEvent;

    stopMessagesProcessing = chatService.watch({
      next: updateEvent => {
        updatesQueue.push(updateEvent);
        if (!updatesProc.getP()) {
          updatesProc.start(processQueuedUpdateEvents);
        }
      },
      complete: () => console.log(`Observation of updates events from ChatService completed.`),
      error: err => console.error(`Error occurred in observation of updates events from ChatService: `, err),
    });
    await refreshChatList();

    const chatsWithVideoCall = await chatService.getChatsWithVideoCall();
    if (size(chatsWithVideoCall) > 0) {
      for (const chatIdObj of chatsWithVideoCall) {
        const chatInd = findIndexOfChatInCurrentList(chatIdObj);
        if (chatInd !== -1) {
          chatList.value[chatInd].callStart = Date.now();
        }
      }
    }

    stopVideoCallsWatching = videoOpenerSrv.watchVideoChats({
      next: data => {
        const { type, chatId } = data;
        const chatIndex = findIndexOfChatInCurrentList(chatId);

        // eslint-disable-next-line default-case
        switch (type) {
          case 'call-started':
            chatList.value[chatIndex].callStart = Date.now();
            break;

          case 'call-ended':
            chatList.value[chatIndex].callStart = undefined;
            break;
        }
      },
      complete: () => console.log('Observation of video call events from VideoOpenerService completed.'),
      error: err => console.error('Error occurred in observation of video call events from VideoOpenerService:', err),
    });
  }

  function stopWatching() {
    stopMessagesProcessing?.();
    stopVideoCallsWatching?.();
  }

  return {
    chatList,
    chatListSortedByTime,

    getChatView,
    refreshChatViewData,
    updateChatItemInList,
    clearIncomingCallsData,
    setIncomingCallsData,

    createNewOneToOneChat,
    createNewGroupChat,
    acceptChatInvitation,

    refreshChatList,
    setChatDialogFlag,

    initialize,
    stopWatching,
  };
});
