"use strict";
/*
 Copyright (C) 2021 3NSoft Inc.

 This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.startWasmFrom = startWasmFrom;
class MP1 {
    constructor(wasmBytes) {
        this.sinkMsgFromWasmInstance = undefined;
        this.msgWaitingWriteCB = undefined;
        const module = new WebAssembly.Module(wasmBytes);
        this.instance = new WebAssembly.Instance(module, this.makeImports());
        this.mp1_accept_msg = this.getExportedFn('_3nweb_mp1_accept_msg');
        Object.seal(this);
    }
    getExportedFn(fName) {
        const fn = this.instance.exports[fName];
        if (typeof fn !== 'function') {
            throw Error(`WASM instance doesn't export function ${fName}`);
        }
        return fn;
    }
    makeImports() {
        const env = {
            _3nweb_mp1_send_out_msg: (ptr, len) => this.processMsgFromModule(ptr, len),
            _3nweb_mp1_write_msg_into: (ptr) => this.writeMsgInCB(ptr),
        };
        return { env };
    }
    processMsgFromModule(ptr, len) {
        if (!this.sinkMsgFromWasmInstance) {
            return;
        }
        const msgBuf = this.getWasmMemoryArea(ptr, len);
        const msg = new Uint8Array(msgBuf.length);
        msg.set(msgBuf);
        this.sinkMsgFromWasmInstance(msg);
    }
    writeMsgInCB(ptr) {
        const buf = this.getWasmMemoryArea(ptr, this.msgWaitingWriteCB.length);
        buf.set(this.msgWaitingWriteCB);
        this.msgWaitingWriteCB = undefined;
    }
    getWasmMemoryArea(ptr, len) {
        const memBuf = this.instance.exports.memory.buffer;
        return new Uint8Array(memBuf, ptr, len);
    }
    start() {
        this.getExportedFn('_start')();
    }
    sendMsg(msg) {
        this.msgWaitingWriteCB = msg;
        // Note that following call to WASM expects it to call back imported
        // function that actually copies bytes from this.msgWaitingWriteCB into
        // then given memory area.
        this.mp1_accept_msg(msg.length);
    }
    setMsgListener(listener) {
        this.sinkMsgFromWasmInstance = listener;
    }
}
Object.freeze(MP1.prototype);
Object.freeze(MP1);
function startWasmFrom(wasmModuleBytes) {
    let mp1 = new MP1(wasmModuleBytes);
    mp1.start();
    return {
        sendMsgIntoWASM: msg => mp1.sendMsg(msg),
        setMsgListener: listener => mp1.setMsgListener(listener)
    };
}
Object.freeze(exports);
