"use strict";
/*
 Copyright (C) 2021, 2025 3NSoft Inc.
 
 This program is free software: you can redistribute it and/or modify it under
 the terms of the GNU General Public License as published by the Free Software
 Foundation, either version 3 of the License, or (at your option) any later
 version.
 
 This program is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License along with
 this program. If not, see <http://www.gnu.org/licenses/>.
*/
Object.defineProperty(exports, "__esModule", { value: true });
const worker_threads_1 = require("worker_threads");
const buffer_utils_1 = require("../../lib-common/buffer-utils");
if (!worker_threads_1.parentPort) {
    throw new Error(`This module can be called as worker's main only`);
}
let { deviceFilePath, bufferWithBytes } = worker_threads_1.workerData;
if (bufferWithBytes) {
    bufferWithBytes = (0, buffer_utils_1.toBuffer)(bufferWithBytes);
}
else if (typeof deviceFilePath !== 'string') {
    throw new Error(`Neither zip file path, nor nor buffer with zip content are given`);
}
const AdmZip = require('adm-zip');
let zip = undefined;
let entries;
function setZip() {
    zip = (new AdmZip(deviceFilePath ? deviceFilePath : bufferWithBytes));
    const lst = zip.getEntries();
    entries = new Map();
    for (const e of lst) {
        entries.set(e.entryName, e);
    }
}
worker_threads_1.parentPort.on('message', (msg) => {
    try {
        if (!zip) {
            setZip();
        }
        let reply;
        if (msg.op === 'list') {
            reply = {
                op: msg.op,
                lst: Array.from(entries.values()).map(e => ({
                    entryName: e.entryName, isDirectory: e.isDirectory
                }))
            };
        }
        else if (msg.op === 'read-file') {
            const entry = entries.get(msg.entryName);
            if (!entry) {
                throw new Error(`Unknown zip entry ${msg.entryName}`);
            }
            reply = {
                op: msg.op,
                bytes: zip.readFile(entry)
            };
        }
        else {
            throw new Error(`Unknown operation in ${JSON.stringify(msg)}`);
        }
        worker_threads_1.parentPort.postMessage(reply);
    }
    catch (err) {
        const reply = {
            op: msg.op,
            err: err.stack
        };
        worker_threads_1.parentPort.postMessage(reply);
    }
});
