/*
 Copyright (C) 2022 - 2024 3NSoft Inc.
 
 This program is free software: you can redistribute it and/or modify it under
 the terms of the GNU General Public License as published by the Free Software
 Foundation, either version 3 of the License, or (at your option) any later
 version.
 
 This program is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License along with
 this program. If not, see <http://www.gnu.org/licenses/>.
*/
import { jsonFromBytes, jsonToBytes, strFromBytes, strToBytes } from './test-page-utils.js';
export class Service {
    syncFS;
    localFS;
    waitMillisBeforeClosing;
    scheduledClosure = undefined;
    constructor(syncFS, localFS, waitMillisBeforeClosing) {
        this.syncFS = syncFS;
        this.localFS = localFS;
        this.waitMillisBeforeClosing = waitMillisBeforeClosing;
        Object.seal(this);
    }
    handleConnection(connection) {
        connection.watch({
            next: async (call) => {
                if (call.msgType === 'start') {
                    await this.serveCallStart(connection, call);
                }
                else if (call.msgType === 'cancel') {
                }
                else {
                    await w3n.testStand.log('error', `Got unknown message type '${call.msgType}'`);
                    w3n.closeSelf();
                }
            },
            complete: () => {
                if (this.waitMillisBeforeClosing === 0) {
                    w3n.closeSelf();
                }
                if (this.scheduledClosure !== undefined) {
                    clearTimeout(this.scheduledClosure);
                }
                this.scheduledClosure = setTimeout(() => w3n.closeSelf(), this.waitMillisBeforeClosing);
            },
            error: async (err) => {
                await w3n.testStand.log('error', `Error in listening for calls`, err);
                w3n.closeSelf();
            }
        });
    }
    static singleton = undefined;
    async serveCallStart(connection, { callNum, method, data }) {
        try {
            if (method === 'foo') {
                this.foo();
                await connection.send({ callNum, callStatus: 'end' });
            }
            else if (method === 'getUniqueIdentifier') {
                const reply = this.getUniqueIdentifier();
                await connection.send({
                    callNum, callStatus: 'end', data: reply
                });
            }
            else if (method === 'addToBytes') {
                const reply = this.addToBytes(data);
                await connection.send({
                    callNum, callStatus: 'end', data: reply
                });
            }
            else if (method === 'writeFileInSyncFS') {
                await this.writeFileInSyncFS(data);
                await connection.send({ callNum, callStatus: 'end' });
            }
            else if (method === 'readFileFromSyncFS') {
                const content = await this.readFileFromSyncFS(data);
                await connection.send({
                    callNum, callStatus: 'end', data: content
                });
            }
            else if (method === 'writeJSONFileInSyncFS') {
                await this.writeJSONFileInSyncFS(data);
                await connection.send({ callNum, callStatus: 'end' });
            }
            else if (method === 'readJSONFileFromSyncFS') {
                const content = await this.readJSONFileFromSyncFS(data);
                await connection.send({
                    callNum, callStatus: 'end', data: content
                });
            }
            else if (method === 'writeFileInLocalFS') {
                await this.writeFileInLocalFS(data);
                await connection.send({ callNum, callStatus: 'end' });
            }
            else if (method === 'readFileFromLocalFS') {
                const content = await this.readFileFromLocalFS(data);
                await connection.send({
                    callNum, callStatus: 'end', data: content
                });
            }
            else if (method === 'writeJSONFileInLocalFS') {
                await this.writeJSONFileInLocalFS(data);
                await connection.send({ callNum, callStatus: 'end' });
            }
            else if (method === 'readJSONFileFromLocalFS') {
                const content = await this.readJSONFileFromLocalFS(data);
                await connection.send({
                    callNum, callStatus: 'end', data: content
                });
            }
            else if (method === 'getUserId') {
                const data = await this.getUserId();
                await connection.send({
                    callNum, callStatus: 'end', data
                });
            }
            else if (method === 'readAndPassFile') {
                const reply = await this.readAndPassFile(data);
                await connection.send({
                    callNum, callStatus: 'end', data: reply
                });
            }
            else if (method === 'readAndPassFS') {
                const reply = await this.readAndPassFS(data);
                await connection.send({
                    callNum, callStatus: 'end', data: reply
                });
            }
            else {
                await connection.send({
                    callNum, callStatus: 'error', err: `Method ${method} not found`
                });
            }
        }
        catch (err) {
            await connection.send({
                callNum, callStatus: 'error', err
            });
        }
    }
    foo() { }
    uid = `${Math.floor(Number.MAX_SAFE_INTEGER * Math.random())}`;
    getUniqueIdentifier() {
        return { bytes: strToBytes(this.uid) };
    }
    addToBytes({ bytes }) {
        for (let i = 0; i < bytes.length; i += 1) {
            bytes[i] += i;
        }
        return { bytes };
    }
    async writeFileInSyncFS({ bytes }) {
        const { path, body } = parsePathAndBody(bytes);
        await this.syncFS.writeBytes(path, body);
    }
    async readFileFromSyncFS({ bytes }) {
        const { path } = parsePathAndBody(bytes);
        const data = await this.syncFS.readBytes(path);
        return (data ? { bytes: data } : undefined);
    }
    async writeFileInLocalFS({ bytes }) {
        const { path, body } = parsePathAndBody(bytes);
        await this.localFS.writeBytes(path, body);
    }
    async readFileFromLocalFS({ bytes }) {
        const { path } = parsePathAndBody(bytes);
        const data = await this.localFS.readBytes(path);
        return (data ? { bytes: data } : undefined);
    }
    async writeJSONFileInSyncFS({ bytes }) {
        const { path, json } = parsePathAndJSONBody(bytes);
        await this.syncFS.writeJSONFile(path, json);
    }
    async readJSONFileFromSyncFS({ bytes }) {
        const { path } = parsePathAndBody(bytes);
        const data = await this.syncFS.readJSONFile(path);
        return { bytes: jsonToBytes(data) };
    }
    async writeJSONFileInLocalFS({ bytes }) {
        const { path, json } = parsePathAndJSONBody(bytes);
        await this.localFS.writeJSONFile(path, json);
    }
    async readJSONFileFromLocalFS({ bytes }) {
        const { path } = parsePathAndBody(bytes);
        const data = await this.localFS.readJSONFile(path);
        return { bytes: jsonToBytes(data) };
    }
    async getUserId() {
        const userId = await w3n.mailerid.getUserId();
        return { bytes: strToBytes(userId) };
    }
    async readAndPassFile({ passedByReference }) {
        const file = passedByReference[0];
        if (file !== passedByReference[1]) {
            throw new Error(`Duplicate reference should produce same file object`);
        }
        const txtContent = await file.readTxt();
        return {
            bytes: strToBytes(txtContent),
            passedByReference: [file, passedByReference[1]]
        };
    }
    async readAndPassFS({ bytes, passedByReference }) {
        const fs = passedByReference[0];
        if (fs !== passedByReference[1]) {
            throw new Error(`Duplicate reference should produce same fs object`);
        }
        const filePath = strFromBytes(bytes);
        const txtContent = await fs.readTxtFile(filePath);
        return {
            bytes: strToBytes(txtContent),
            passedByReference: [fs, passedByReference[1]]
        };
    }
}
function parsePathAndBody(bytes) {
    const pathLen = bytes[0];
    const path = strFromBytes(bytes.slice(1, pathLen + 1));
    const body = bytes.slice(pathLen + 1);
    return { path, body };
}
function parsePathAndJSONBody(bytes) {
    const { path, body } = parsePathAndBody(bytes);
    return { path, json: jsonFromBytes(body) };
}
//# sourceMappingURL=service.js.map