class Service {
    syncFS;
    localFS;
    waitMillisBeforeClosing;
    scheduledClosure;
    constructor(syncFS, localFS, waitMillisBeforeClosing){
        this.syncFS = syncFS;
        this.localFS = localFS;
        this.waitMillisBeforeClosing = waitMillisBeforeClosing;
        this.scheduledClosure = undefined;
        this.uid = `${Math.floor(Number.MAX_SAFE_INTEGER * Math.random())}`;
        Object.seal(this);
    }
    handleConnection(connection) {
        connection.watch({
            next: async (call)=>{
                if (call.msgType === 'start') {
                    await this.serveCallStart(connection, call);
                } else if (call.msgType === 'cancel') {} else {
                    await w3n.testStand.log('error', `Got unknown message type '${call.msgType}'`);
                    w3n.closeSelf();
                }
            },
            complete: ()=>{
                if (this.waitMillisBeforeClosing === 0) {
                    w3n.closeSelf();
                }
                if (this.scheduledClosure !== undefined) {
                    clearTimeout(this.scheduledClosure);
                }
                this.scheduledClosure = setTimeout(()=>w3n.closeSelf(), this.waitMillisBeforeClosing);
            },
            error: async (err)=>{
                await w3n.testStand.log('error', `Error in listening for calls`, err);
                w3n.closeSelf();
            }
        });
    }
    static singleton = undefined;
    async serveCallStart(connection, { callNum, method, data }) {
        try {
            if (method === 'foo') {
                this.foo();
                await connection.send({
                    callNum,
                    callStatus: 'end'
                });
            } else if (method === 'getUniqueIdentifier') {
                const reply = this.getUniqueIdentifier();
                await connection.send({
                    callNum,
                    callStatus: 'end',
                    data: reply
                });
            } else if (method === 'addToBytes') {
                const reply = this.addToBytes(data);
                await connection.send({
                    callNum,
                    callStatus: 'end',
                    data: reply
                });
            } else if (method === 'writeFileInSyncFS') {
                await this.writeFileInSyncFS(data);
                await connection.send({
                    callNum,
                    callStatus: 'end'
                });
            } else if (method === 'readFileFromSyncFS') {
                const content = await this.readFileFromSyncFS(data);
                await connection.send({
                    callNum,
                    callStatus: 'end',
                    data: content
                });
            } else if (method === 'writeJSONFileInSyncFS') {
                await this.writeJSONFileInSyncFS(data);
                await connection.send({
                    callNum,
                    callStatus: 'end'
                });
            } else if (method === 'readJSONFileFromSyncFS') {
                const content = await this.readJSONFileFromSyncFS(data);
                await connection.send({
                    callNum,
                    callStatus: 'end',
                    data: content
                });
            } else if (method === 'writeFileInLocalFS') {
                await this.writeFileInLocalFS(data);
                await connection.send({
                    callNum,
                    callStatus: 'end'
                });
            } else if (method === 'readFileFromLocalFS') {
                const content = await this.readFileFromLocalFS(data);
                await connection.send({
                    callNum,
                    callStatus: 'end',
                    data: content
                });
            } else if (method === 'writeJSONFileInLocalFS') {
                await this.writeJSONFileInLocalFS(data);
                await connection.send({
                    callNum,
                    callStatus: 'end'
                });
            } else if (method === 'readJSONFileFromLocalFS') {
                const content = await this.readJSONFileFromLocalFS(data);
                await connection.send({
                    callNum,
                    callStatus: 'end',
                    data: content
                });
            } else if (method === 'getUserId') {
                const data = await this.getUserId();
                await connection.send({
                    callNum,
                    callStatus: 'end',
                    data
                });
            } else {
                await connection.send({
                    callNum,
                    callStatus: 'error',
                    err: `Method ${method} not found`
                });
            }
        } catch (err) {
            await connection.send({
                callNum,
                callStatus: 'error',
                err
            });
        }
    }
    foo() {}
    uid;
    getUniqueIdentifier() {
        return {
            bytes: strToBytes(this.uid)
        };
    }
    addToBytes({ bytes }) {
        for(let i = 0; i < bytes.length; i += 1){
            bytes[i] += i;
        }
        return {
            bytes
        };
    }
    async writeFileInSyncFS({ bytes }) {
        const { path, body } = parsePathAndBody(bytes);
        await this.syncFS.writeBytes(path, body);
    }
    async readFileFromSyncFS({ bytes }) {
        const { path } = parsePathAndBody(bytes);
        const data = await this.syncFS.readBytes(path);
        return data ? {
            bytes: data
        } : undefined;
    }
    async writeFileInLocalFS({ bytes }) {
        const { path, body } = parsePathAndBody(bytes);
        await this.localFS.writeBytes(path, body);
    }
    async readFileFromLocalFS({ bytes }) {
        const { path } = parsePathAndBody(bytes);
        const data = await this.localFS.readBytes(path);
        return data ? {
            bytes: data
        } : undefined;
    }
    async writeJSONFileInSyncFS({ bytes }) {
        const { path, json } = parsePathAndJSONBody(bytes);
        await this.syncFS.writeJSONFile(path, json);
    }
    async readJSONFileFromSyncFS({ bytes }) {
        const { path } = parsePathAndBody(bytes);
        const data = await this.syncFS.readJSONFile(path);
        return {
            bytes: jsonToBytes(data)
        };
    }
    async writeJSONFileInLocalFS({ bytes }) {
        const { path, json } = parsePathAndJSONBody(bytes);
        await this.localFS.writeJSONFile(path, json);
    }
    async readJSONFileFromLocalFS({ bytes }) {
        const { path } = parsePathAndBody(bytes);
        const data = await this.localFS.readJSONFile(path);
        return {
            bytes: jsonToBytes(data)
        };
    }
    async getUserId() {
        const userId = await w3n.mailerid.getUserId();
        return {
            bytes: strToBytes(userId)
        };
    }
}
function parsePathAndBody(bytes) {
    const pathLen = bytes[0];
    const path = strFromBytes(bytes.slice(1, pathLen + 1));
    const body = bytes.slice(pathLen + 1);
    return {
        path,
        body
    };
}
function parsePathAndJSONBody(bytes) {
    const { path, body } = parsePathAndBody(bytes);
    return {
        path,
        json: jsonFromBytes(body)
    };
}
function jsonFromBytes(bytes) {
    return JSON.parse(strFromBytes(bytes));
}
function jsonToBytes(json) {
    return strToBytes(JSON.stringify(json));
}
function strToBytes(str) {
    const enc = new TextEncoder();
    return enc.encode(str);
}
function strFromBytes(bytes) {
    const dec = new TextDecoder();
    return dec.decode(bytes);
}
function sleep(millis) {
    return new Promise((resolve)=>{
        setTimeout(resolve, millis);
    });
}
const longNonGuiSrvInThisApp = 'LongServiceInDeno';
const syncFS = await w3n.storage.getAppSyncedFS();
const localFS = await w3n.storage.getAppLocalFS();
Service.singleton = new Service(syncFS, localFS, 5000);
await sleep(100);
w3n.rpc.exposeService(longNonGuiSrvInThisApp, {
    next: async (connection)=>{
        Service.singleton.handleConnection(connection);
    },
    complete: ()=>w3n.closeSelf(),
    error: async (err)=>{
        await w3n.testStand.log('error', `Error in listening for incoming connections`, err);
        w3n.closeSelf();
    }
});

