/*
 Copyright (C) 2021 - 2022, 2024 3NSoft Inc.
 
 This program is free software: you can redistribute it and/or modify it under
 the terms of the GNU General Public License as published by the Free Software
 Foundation, either version 3 of the License, or (at your option) any later
 version.
 
 This program is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 See the GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License along with
 this program. If not, see <http://www.gnu.org/licenses/>.
*/
import { getOneMsgFromProcess } from "../libs-for-tests/proc-messaging.js";
import { logErr } from "../../test-page-utils.js";
async function sendTestSignal(userNum, msg) {
    await w3n.testStand.sendMsgToOtherLocalTestProcess(userNum, undefined, undefined, msg);
}
async function echoIncomingMessageToFstUser(msg) {
    if (msg.attachments) {
        try {
            msg.attachments = await listFileTree(msg.attachments);
        }
        catch (err) {
            await w3n.testStand.log('error', `Error in listing incoming message attachments`, err);
        }
    }
    await sendTestSignal(1, {
        testSignal: 'message-echo',
        msg
    });
}
async function listFileTree(fs) {
    const tree = {};
    for (const entry of (await fs.listFolder('.'))) {
        if (entry.isFile) {
            const fileContent = await fs.readBytes(entry.name);
            tree[entry.name] = (fileContent ? fileContent.length : 0);
        }
        else if (entry.isFolder) {
            const subFS = await fs.readonlySubRoot(entry.name);
            tree[entry.name] = await listFileTree(subFS);
        }
    }
    return tree;
}
export async function setupSecondUserASMailTestReactions() {
    const alreadyEchoedMsgs = new Set();
    const userOne = await w3n.testStand.idOfTestUser(1);
    // echo back all mail messages
    w3n.mail.inbox.subscribe('message', {
        next: async (msg) => {
            if (msg.msgType === 'mail') {
                if (alreadyEchoedMsgs.has(msg.msgId)) {
                    await w3n.testStand.log('error', `Inbox generated 'message' event with an already echoed msg ${msg.msgId}, why?`);
                }
                else {
                    alreadyEchoedMsgs.add(msg.msgId);
                    await echoIncomingMessageToFstUser(msg);
                }
            }
        },
        error: err => logErr(`Error occurred in listening for messages`, err)
    });
    // attend signal asking to send message back
    w3n.testStand.observeMsgsFromOtherLocalTestProcess({
        next: async (sig) => {
            if (sig.testSignal !== 'ask-to-send-msg-back') {
                return;
            }
            try {
                await sendMsg(userOne, sig.msg);
            }
            catch (err) {
                await w3n.testStand.log('error', `Error in sending message to ${userOne}`, err);
            }
        }
    }, 1, undefined, undefined);
}
async function sendMsg(userId, msg) {
    const deliveryId = `${Date.now()}`;
    await w3n.mail.delivery.addMsg([userId], msg, deliveryId);
    await new Promise((resolve, reject) => w3n.mail.delivery.observeDelivery(deliveryId, {
        next: async (progress) => {
            if (progress.allDone) {
                const err = progress.recipients[userId].err;
                if (err) {
                    reject(err);
                }
                else {
                    try {
                        await w3n.mail.delivery.rmMsg(deliveryId);
                    }
                    catch (err) {
                        reject(err);
                    }
                    finally {
                        resolve(undefined);
                    }
                }
            }
        },
        error: reject
    }));
}
export function listenForOneMsgEchoFromSecondUser(timeout) {
    return getOneMsgFromProcess(2, undefined, undefined, timeout);
}
export async function askSecondUserToSendMsg(msg) {
    await sendTestSignal(2, {
        testSignal: 'ask-to-send-msg-back',
        msg
    });
}
//# sourceMappingURL=second-user.js.map